package edu.uky.ai.lp.ai;

/**
 * <p>A continuation, sometimes called a generator, is like a function which
 * can return multiple values in sequence.  Each time the continuation returns
 * a value, it pauses execution and waits to be called again.  Subsequent calls
 * will pick up where the last one left off.</p>
 * 
 * <p>A continuation has 4 important types of events in its lifecycle:</p>
 * <ul>
 * <li><tt>CALL</tt>: The first time the continuation is invoked.</li>
 * <li><tt>REDO</tt>: Subsequent invocations.</li>
 * <li><tt>EXIT</tt>: Each time the continuation returns a value.</li>
 * <li><tt>FAIL</tt>: The final event, indicating there are no more values to
 * be returned.</li>
 * </ul>
 * 
 * <p>Continuations are implemented by extending this class and overriding the
 * {@link #run()} method.  Each time the continuation should return a value,
 * {@link #run()} should call {@link #yield(Object)}.  When {@link #run()}
 * finishes, the continuation will FAIL.</p>
 * 
 * @author Stephen G. Ware
 * @param <T> the type of value that this continuation will return
 */
public abstract class Continuation<T> {

	/** The number of times to indent a message before displaying it */
	private static int indent = 0;
	
	/**
	 * Returns a string that can be used to indent a message.
	 * 
	 * @return an indent string
	 */
	private static final String indent() {
		String str = "";
		for(int i=0; i<indent; i++)
			str += "| ";
		return str;
	}
	
	/** The result to be returned to the waiting caller */
	private T result = null;
	
	/** Indicates if the continuation has finished returning values */
	private boolean done = false;
	
	/** The thread on which the continuation runs
	 * (which is also used to preserve its state between invocations) */
	private Thread thread = new Thread() {
		@Override
		public void run() {
			synchronized(Continuation.this) {
				Continuation.this.run();
				indent--;
				System.out.println(indent() + "FAIL " + Continuation.this);
				done = true;
				Continuation.this.notify();
			}
		}
	};
	
	/**
	 * Indicates whether or not more CALLs can be made to this continuation.
	 * 
	 * @return true if no more calls can be mode, false otherwise
	 */
	public boolean done() {
		return done;
	}
	
	/**
	 * Invokes the continuation, picking up where the previous invocation (if
	 * any) stopped.
	 * 
	 * @return the next value returned by the continuation, or null if this call resulted in FAIL
	 */
	public T call() {
		if(done)
			throw new IllegalStateException("Continuation is done");
		synchronized(this) {
			if(thread.getState() == Thread.State.NEW) {
				System.out.println(indent() + "CALL " + this);
				indent++;
				thread.setDaemon(true);
				thread.start();
			}
			else {
				System.out.println(indent() + "REDO " + this);
				indent++;
				notify();
			}
			try { wait(); }
			catch(InterruptedException ex){/* do nothing */}
			if(done)
				return null;
			else
				return result;
		}
	}
	
	/**
	 * The actual work to be done by the continuation.
	 */
	protected abstract void run();
	
	/**
	 * This method is called each time the continuation should return a value.
	 * The value to be returned is passed as a parameter.  When this method is
	 * called, the continuation will pause until the next REDO event.
	 * 
	 * @param value the value to be returned
	 */
	protected final void yield(T value) {
		result = value;
		indent--;
		System.out.println(indent() + "EXIT " + this + " = " + result);
		notify();
		try { wait(); }
		catch(InterruptedException ex){/* do nothing */}
	}
}
